// Unified Notification Service - Sends to Web (PWA), Mobile (Expo), and Socket.IO
const webpush = require('web-push');
const { Expo } = require('expo-server-sdk');
const axios = require('axios');

// Initialize Expo SDK
const expo = new Expo();

// Global io instance (will be set by server.js)
let globalIO = null;

function setIO(io) {
  globalIO = io;
}

// Database helper - promisify callbacks
function queryAsync(query, params = []) {
  return new Promise((resolve, reject) => {
    const mysql = require('../utils/database');
    mysql.query(query, params, (err, results) => {
      if (err) reject(err);
      else resolve(results);
    });
  });
}

// VAPID Keys configuration
const VAPID_PUBLIC_KEY =
  process.env.NEXT_PUBLIC_VAPID_PUBLIC_KEY ||
  'BBVPcdOTcge5pKMiWxgWfBEm2ugpF-NZ6soK9l0bKTpMoXuaShylZcZwor43CYhG4YzOgHuCvnqwM9Fd0wTKLp4';

const VAPID_PRIVATE_KEY =
  process.env.VAPID_PRIVATE_KEY ||
  'EFjxKzqZ6FPxkb0ctu_sBkAxm40A1ymjAdgHkChTZq4';

webpush.setVapidDetails(
  'mailto:engr.basitofficial@gmail.com',
  VAPID_PUBLIC_KEY,
  VAPID_PRIVATE_KEY
);

/**
 * Send notification through all available channels
 * Supports: Socket.IO (real-time), Web Push (PWA), Expo Push (Mobile), SMS
 */
class UnifiedNotificationService {
  /**
   * Send lead notification to a user
   */
  static async sendLeadNotification(userId, username, leadData) {
    try {
      console.log(`📢 Sending lead notification to user ${userId} (${username})`);

      const notificationData = {
        title: 'New Lead Assigned',
        body: `Lead #${leadData.id}: ${leadData.name || leadData.customer_name}`,
        leadId: leadData.id,
        type: 'lead',
      };

      // Send through all channels in parallel
      const results = await Promise.allSettled([
        this.sendSocketIONotification(userId, notificationData),
        this.sendWebPushNotification(username, notificationData),
        this.sendExpoNotification(userId, notificationData),
      ]);

      console.log('✅ Lead notification sent to all channels');
      return { success: true, results };
    } catch (error) {
      console.error('❌ Error sending lead notification:', error.message);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send follow-up reminder notification
   */
  static async sendFollowUpNotification(userId, username, followUpData) {
    try {
      console.log(`📢 Sending follow-up notification to user ${userId} (${username})`);

      const notificationData = {
        title: 'Follow-Up Reminder',
        body: `Reminder: ${followUpData.notes || 'Follow up with lead'}`,
        leadId: followUpData.lead_id,
        type: 'followup',
        scheduledTime: followUpData.scheduled_time,
      };

      const results = await Promise.allSettled([
        this.sendSocketIONotification(userId, notificationData),
        this.sendWebPushNotification(username, notificationData),
        this.sendExpoNotification(userId, notificationData),
      ]);

      console.log('✅ Follow-up notification sent');
      return { success: true, results };
    } catch (error) {
      console.error('❌ Error sending follow-up notification:', error.message);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send real-time notification via Socket.IO
   */
  static async sendSocketIONotification(userId, notificationData) {
    try {
      if (!globalIO) {
        console.warn('⚠️ Socket.IO not initialized');
        return { success: false, message: 'Socket.IO not initialized' };
      }

      const roomName = `user_${userId}`;
      globalIO.to(roomName).emit('notification', {
        ...notificationData,
        timestamp: new Date().toISOString(),
        channel: 'socket.io',
      });

      console.log(`✅ Socket.IO notification sent to room: ${roomName}`);

      // Log to database
      await this.logNotification(userId, null, notificationData, 'realtime');

      return { success: true, message: 'Socket.IO notification sent' };
    } catch (error) {
      console.error('❌ Socket.IO error:', error.message);
      return { success: false, error: error.message };
    }
  }

  /**
   * Send Web Push notification (PWA)
   */
  static async sendWebPushNotification(username, notificationData) {
    try {
      const rows = await queryAsync('SELECT subscription FROM lead_subscription WHERE username = ?', [username]);

      if (rows.length === 0) {
        console.warn(`⚠️ No web push subscription found for user: ${username}`);
        return { success: false, message: 'No subscription found' };
      }

      const subscription = JSON.parse(rows[0].subscription);

      if (!subscription?.endpoint) {
        console.error('❌ Invalid subscription object');
        return { success: false, message: 'Invalid subscription' };
      }

      const payload = JSON.stringify({
        title: notificationData.title,
        body: notificationData.body,
        icon: '/icons/icon-192x192.png',
        badge: '/icons/icon-96x96.png',
        tag: `notification-${notificationData.leadId}`,
        requireInteraction: false,
        data: {
          url:
            notificationData.type === 'lead'
              ? `/leads/${notificationData.leadId}`
              : `/leads`,
          leadId: notificationData.leadId || null,
          type: notificationData.type,
        },
      });

      await webpush.sendNotification(subscription, payload);
      console.log(`✅ Web push notification sent to ${username}`);

      // Log to database
      await this.logNotification(null, username, notificationData, 'web');

      return { success: true, message: 'Web push sent' };
    } catch (error) {
      console.error('❌ Web push error:', error.message);

      // Remove invalid subscriptions
      if (error?.statusCode === 404 || error?.statusCode === 410) {
        await queryAsync('DELETE FROM lead_subscription WHERE username = ?', [username]);
        console.log(`🗑️ Removed expired subscription for ${username}`);
      }

      return { success: false, error: error.message };
    }
  }

  /**
   * Send Expo Push notification (React Native)
   */
  static async sendExpoNotification(userId, notificationData) {
    try {
      const rows = await queryAsync('SELECT token FROM expo_push_tokens WHERE user_id = ?', [userId]);

      if (rows.length === 0) {
        console.warn(`⚠️ No Expo push token found for user ${userId}`);
        return { success: false, message: 'No token found' };
      }

      const pushToken = rows[0].token;

      if (!Expo.isExpoPushToken(pushToken)) {
        console.error(`❌ Invalid Expo token for user ${userId}`);
        await queryAsync('DELETE FROM expo_push_tokens WHERE user_id = ?', [userId]);
        return { success: false, message: 'Invalid token' };
      }

      const message = {
        to: pushToken,
        sound: 'default',
        title: notificationData.title,
        body: notificationData.body,
        data: {
          leadId: notificationData.leadId,
          type: notificationData.type,
          url:
            notificationData.type === 'lead'
              ? `/leads/${notificationData.leadId}`
              : `/leads`,
        },
        priority: 'high',
        channelId: 'default',
      };

      const chunks = expo.chunkPushNotifications([message]);
      const tickets = [];

      for (const chunk of chunks) {
        try {
          const ticketChunk = await expo.sendPushNotificationsAsync(chunk);
          tickets.push(...ticketChunk);
        } catch (error) {
          console.error('❌ Error sending push notification chunk:', error);
        }
      }

      for (const ticket of tickets) {
        if (ticket.status === 'error') {
          console.error(`❌ Expo error: ${ticket.message}`);
          if (ticket.details?.error === 'DeviceNotRegistered') {
            await queryAsync('DELETE FROM expo_push_tokens WHERE user_id = ?', [userId]);
          }
          return { success: false, error: ticket.message };
        }
      }

      console.log(`✅ Expo push notification sent to user ${userId}`);

      // Log to database
      await this.logNotification(userId, null, notificationData, 'mobile');

      return { success: true, message: 'Expo push sent' };
    } catch (error) {
      console.error('❌ Expo push error:', error.message);
      return { success: false, error: error.message };
    }
  }

  /**
   * Log notification to database for tracking
   */
  static async logNotification(userId, username, notificationData, notificationType) {
    try {
      const query = `
        INSERT INTO real_time_notifications 
        (user_id, username, title, body, lead_id, notification_type, created_at)
        VALUES (?, ?, ?, ?, ?, ?, NOW())
      `;

      await queryAsync(query, [
        userId || null,
        username || null,
        notificationData.title,
        notificationData.body,
        notificationData.leadId || null,
        notificationType,
      ]);
    } catch (error) {
      console.error('❌ Error logging notification:', error.message);
    }
  }

  /**
   * Get notification history for a user
   */
  static async getNotificationHistory(userId, limit = 20) {
    try {
      const query = `
        SELECT * FROM real_time_notifications 
        WHERE user_id = ? OR username = ?
        ORDER BY created_at DESC
        LIMIT ?
      `;

      const notifications = await queryAsync(query, [userId, null, limit]);

      return { success: true, data: notifications };
    } catch (error) {
      console.error('❌ Error fetching notification history:', error.message);
      return { success: false, error: error.message };
    }
  }
}

module.exports = UnifiedNotificationService;
module.exports.setIO = setIO;
