const { z } = require('zod');
const crypto = require('crypto');
const mysqlConnection = require('../utils/database');
const fs = require('fs');
const { permission } = require('process');
const { generateToken } = require('../middleware/jwtMiddleware');


const loginHandler = async (req, res) => {
  try {
    let parsedBody;
    try {
      parsedBody = req.body && req.body.body ? JSON.parse(req.body.body) : req.body;
    } catch (err) {
      console.error('JSON parse error in loginHandler:', err);
      return res.status(400).json({ success: false, message: 'Invalid request body' });
    }

    const { email, password } = parsedBody;

    if (!email || !password) {
      return res.status(400).json({ success: false, message: 'Invalid email or password' });
    }

    // join users_types to pull permission fields
    const [rows] = await mysqlConnection.promise().query(
      `SELECT 
         u.id AS user_id, u.email, u.name, u.user_type AS user_type_key, u.company_id, u.password, u.files_1,
         ut.id AS type_id, ut.type AS type_name, ut.permission_level, ut.del AS type_del,
         ut.Create_permission, ut.Edit_permission, ut.View_permission
       FROM users u
       JOIN users_types ut ON u.user_type = ut.type
       WHERE u.email = ? AND u.del = "N"`,
      [email]
    );

    if (!rows.length) {
      return res.status(404).json({ success: false, message: 'User Not Found' });
    }

    const dbRow = rows[0];
    const storedPassword = dbRow.password;
    const hashedInput = crypto.createHash('sha256').update(password).digest('hex');

    if (hashedInput !== storedPassword) {
      return res.status(401).json({ success: false, message: 'Incorrect password' });
    }

    // build permissions object from users_types
    const permissions = {
      permission_level: dbRow.permission_level,
      Create_permission: dbRow.Create_permission,
      Edit_permission: dbRow.Edit_permission,
      View_permission: dbRow.View_permission,
    };

    // build user_type detail
    const userType = {
      id: dbRow.type_id,
      key: dbRow.type_name,
      del: dbRow.type_del,
    };

    const userData = {
      id: dbRow.user_id,
      email: dbRow.email,
      name: dbRow.name,
      user_type: dbRow.user_type_key, // original user_type value on users table
      user_type_detail: userType,     // detailed fields from users_types table
      company_id: dbRow.company_id,
      permissions,
      profileImage: dbRow.files_1 || null, // Include profile image from files_1 column
    };

    // generate jwt with full payload
    const token = generateToken(userData);

    // set cookie (keep for web; mobile uses returned token)
    res.cookie('token', token, {
      httpOnly: true,
      secure: process.env.NODE_ENV === 'production',
      sameSite: process.env.NODE_ENV === 'production' ? 'none' : 'lax',
      maxAge: 24 * 60 * 60 * 1000,
      path: '/',
    });

    return res.status(200).json({
      success: true,
      message: 'Login successful',
      user: userData,
      token,
    });
  } catch (error) {
    console.error('500 Error during login:', error);
    return res.status(500).json({ success: false, message: 'Error in login', error: error.message });
  }
};

Checkuserauth = async (req, res) => {
  try {
    if (!req.user || !req.user.id) {
      return res.status(401).json({ success: false, message: 'Invalid token' });
    }

    // Return token payload (fast). It already contains user and permissions.
    return res.status(200).json({
      success: true,
      user: {
        id: req.user.id,
        email: req.user.email,
        name: req.user.name,
        user_type: req.user.user_type,
        user_type_detail: req.user.user_type_detail || {},
        company_id: req.user.company_id,
        permissions: req.user.permissions || {},
        profileImage: req.user.profileImage || null,
      },
    });
  } catch (error) {
    console.error('Error in Checkuserauth:', error);
    return res.status(500).json({ success: false, message: 'Error fetching user', error: error.message });
  }
}


const checkSupperAdmin = async (req, res) => {
  try {
    const { email } = req.params;

    // Fetch user details and company_creator from tables
    const [user] = await mysqlConnection.promise().query('SELECT u.user_type, c.company_creator FROM users u INNER JOIN companies c ON u.name = c.company_creator WHERE u.email = ?', [email]);

    if (!user.length) {
      return res.status(200).json({
        success: true,
        message: 'No company registered',
        user: "not_superadmin"
      });
    } else {
      return res.status(200).json({
        success: true,
        message: 'Company registered',
        user: {
          user_type: user[0].user_type,
          company_creator: user[0].company_creator
        }
      });
    }
  } catch (error) {
    console.error('Error in checkSupperAdmin:', error);
    return res.status(500).json({
      success: false,
      message: 'Error in checking super admin',
      error: error.message
    });
  }
};


const loginHandlerA = async (req, res) => {
  try {
    const parsedBody = JSON.parse(req.body.body); // Why parse body.body? Should be req.body directly
    const { email, password } = parsedBody;

    if (!email || !password) {
      return res.status(400).json({
        success: false,
        message: 'Invalid email or password',
      });
    }

    const [user] = await mysqlConnection.promise().query('SELECT * FROM users WHERE email = ?', [email]);
   const inputToCheck= password
    if (!user.length) {
      return res.status(404).json({
        success: false,
        message: 'Email is not registered',
      });
    }

    // Security issue: You're using SHA-256 for password hashing (not secure for passwords)
    const hashedInput = crypto.createHash('sha256').update(password).digest('hex');
    
    if (hashedInput !== user[0].password) { // Use !== for strict comparison
      return res.status(401).json({
        success: false,
        message: 'Incorrect password',
      });
    }

    // Typo in View_permissio (missing 'n')
    const [perm] = await mysqlConnection.promise().query(`
      SELECT ut.permission_level AS permission, ut.Create_permission, ut.Edit_permission, ut.View_permission
      FROM users_types ut 
      INNER JOIN users u ON u.user_type = ut.type WHERE u.id=?`, [user[0].id]); // Fixed id reference

    const responseData = {
      success: true,
      message: 'Login successful',
      user: {
        email: user[0].email,
        id: user[0].id,
        password: inputToCheck,
        name: `${user[0].first_name} ${user[0].last_name}`,
        username: user[0].name,
        user_type: user[0].user_type,
        company_id: user[0].company_id,
        permission: perm[0]?.permission || 0, // Safe access with fallback
        Create_permission: perm[0]?.Create_permission || 0,
        Edit_permission: perm[0]?.Edit_permission || 0,
        View_permission: perm[0]?.View_permission || 0 // Fixed typo
      }
    };

    res.status(200).json(responseData);

  } catch (error) {
    console.error('Error during login:', error);
    res.status(500).json({
      success: false,
      message: 'Error in login',
      error: error.message,
    });
  }
};


module.exports = { loginHandler, loginHandlerA, checkSupperAdmin, Checkuserauth };
