// Controller for Expo Push Notifications (React Native)
const mysqlConnection = require('../../utils/database');
const { Expo } = require('expo-server-sdk');

// Create a new Expo SDK client
const expo = new Expo();

/**
 * Save or update Expo push token for a user
 */
const saveExpoPushToken = async (req, res) => {
    try {
        const { userId, token } = req.body;

        if (!userId || !token) {
            return res.status(400).json({
                success: false,
                message: 'User ID and token are required',
            });
        }

        // Validate the token format
        if (!Expo.isExpoPushToken(token)) {
            return res.status(400).json({
                success: false,
                message: 'Invalid Expo push token format',
            });
        }

        // Check if token already exists for this user
        const checkQuery = 'SELECT * FROM expo_push_tokens WHERE user_id = ?';
        const [existing] = await mysqlConnection.promise().query(checkQuery, [userId]);

        if (existing.length > 0) {
            // Update existing token
            const updateQuery = 'UPDATE expo_push_tokens SET token = ?, updated_at = NOW() WHERE user_id = ?';
            await mysqlConnection.promise().query(updateQuery, [token, userId]);
        } else {
            // Insert new token
            const insertQuery = 'INSERT INTO expo_push_tokens (user_id, token) VALUES (?, ?)';
            await mysqlConnection.promise().query(insertQuery, [userId, token]);
        }

        return res.status(200).json({
            success: true,
            message: 'Expo push token saved successfully',
        });
    } catch (error) {
        console.error('Error saving Expo push token:', error);
        return res.status(500).json({
            success: false,
            message: 'Failed to save Expo push token',
            error: error.message,
        });
    }
};

/**
 * Delete Expo push token for a user (on logout)
 */
const deleteExpoPushToken = async (req, res) => {
    try {
        const { userId } = req.params;

        if (!userId) {
            return res.status(400).json({
                success: false,
                message: 'User ID is required',
            });
        }

        const deleteQuery = 'DELETE FROM expo_push_tokens WHERE user_id = ?';
        const [result] = await mysqlConnection.promise().query(deleteQuery, [userId]);

        if (result.affectedRows === 0) {
            return res.status(404).json({
                success: false,
                message: 'No token found for this user',
            });
        }

        return res.status(200).json({
            success: true,
            message: 'Expo push token deleted successfully',
        });
    } catch (error) {
        console.error('Error deleting Expo push token:', error);
        return res.status(500).json({
            success: false,
            message: 'Failed to delete Expo push token',
            error: error.message,
        });
    }
};

/**
 * Send Expo push notification to a specific user
 */
const sendExpoNotification = async (userId, title, body, data = {}) => {
    try {
        // Get user's Expo push token
        const query = 'SELECT token FROM expo_push_tokens WHERE user_id = ?';
        const [rows] = await mysqlConnection.promise().query(query, [userId]);

        if (rows.length === 0) {
            console.log(`No Expo push token found for user ${userId}`);
            return false;
        }

        const pushToken = rows[0].token;

        // Validate token
        if (!Expo.isExpoPushToken(pushToken)) {
            console.error(`Invalid Expo push token for user ${userId}:`, pushToken);
            // Clean up invalid token
            await mysqlConnection.promise().query('DELETE FROM expo_push_tokens WHERE user_id = ?', [userId]);
            return false;
        }

        // Create the message
        const message = {
            to: pushToken,
            sound: 'default',
            title: title,
            body: body,
            data: data,
            priority: 'high',
            channelId: 'default',
        };

        // Send the notification
        const chunks = expo.chunkPushNotifications([message]);
        const tickets = [];

        for (const chunk of chunks) {
            try {
                const ticketChunk = await expo.sendPushNotificationsAsync(chunk);
                tickets.push(...ticketChunk);
            } catch (error) {
                console.error('Error sending push notification chunk:', error);
            }
        }

        // Check for errors in tickets
        for (const ticket of tickets) {
            if (ticket.status === 'error') {
                console.error('Expo push notification error:', ticket.message);

                // If token is invalid, remove it from database
                if (ticket.details?.error === 'DeviceNotRegistered') {
                    await mysqlConnection.promise().query('DELETE FROM expo_push_tokens WHERE user_id = ?', [userId]);
                }
            }
        }

        console.log(`Expo push notification sent to user ${userId}`);
        return true;
    } catch (error) {
        console.error('Error sending Expo push notification:', error);
        return false;
    }
};

/**
 * Send Expo push notifications to multiple users
 */
const sendExpoNotificationToMultiple = async (userIds, title, body, data = {}) => {
    try {
        if (!Array.isArray(userIds) || userIds.length === 0) {
            return { success: false, message: 'No user IDs provided' };
        }

        // Get all tokens for the users
        const placeholders = userIds.map(() => '?').join(',');
        const query = `SELECT user_id, token FROM expo_push_tokens WHERE user_id IN (${placeholders})`;
        const [rows] = await mysqlConnection.promise().query(query, userIds);

        if (rows.length === 0) {
            console.log('No Expo push tokens found for provided users');
            return { success: false, message: 'No tokens found' };
        }

        // Create messages for all valid tokens
        const messages = [];
        for (const row of rows) {
            if (Expo.isExpoPushToken(row.token)) {
                messages.push({
                    to: row.token,
                    sound: 'default',
                    title: title,
                    body: body,
                    data: { ...data, userId: row.user_id },
                    priority: 'high',
                    channelId: 'default',
                });
            }
        }

        if (messages.length === 0) {
            return { success: false, message: 'No valid tokens found' };
        }

        // Send notifications in chunks
        const chunks = expo.chunkPushNotifications(messages);
        const tickets = [];

        for (const chunk of chunks) {
            try {
                const ticketChunk = await expo.sendPushNotificationsAsync(chunk);
                tickets.push(...ticketChunk);
            } catch (error) {
                console.error('Error sending push notification chunk:', error);
            }
        }

        console.log(`Sent ${tickets.length} Expo push notifications`);
        return { success: true, ticketCount: tickets.length };
    } catch (error) {
        console.error('Error sending Expo push notifications to multiple users:', error);
        return { success: false, error: error.message };
    }
};

/**
 * Clean up expired or invalid tokens
 */
const cleanupExpiredTokens = async () => {
    try {
        // Delete tokens that haven't been updated in 90 days
        const query = 'DELETE FROM expo_push_tokens WHERE updated_at < DATE_SUB(NOW(), INTERVAL 90 DAY)';
        const [result] = await mysqlConnection.promise().query(query);

        console.log(`Cleaned up ${result.affectedRows} expired Expo push tokens`);
        return result.affectedRows;
    } catch (error) {
        console.error('Error cleaning up expired tokens:', error);
        return 0;
    }
};

module.exports = {
    saveExpoPushToken,
    deleteExpoPushToken,
    sendExpoNotification,
    sendExpoNotificationToMultiple,
    cleanupExpiredTokens,
};
